/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package demo.jdbc.visitor.tests;

import org.apache.olingo.odata2.api.edm.Edm;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.rt.RuntimeDelegate;
import org.apache.olingo.odata2.api.uri.UriParser;
import org.apache.olingo.odata2.api.uri.expression.FilterExpression;
import org.junit.Test;

import demo.jdbc.visitor.JdbcSimpleStringVisitor;
import demo.jdbc.visitor.PropertyNotFoundException;
import demo.jdbc.visitor.TestEdmProvider;

public class JdbcSimpleStringVisitorTest {

  @Test
  public void printExpression() throws Exception {
    String rawExpression = "'a' eq 'b'";
    FilterExpression expression = UriParser.parseFilter(null, null, rawExpression);
    String whereClause = (String) expression.accept(new JdbcSimpleStringVisitor());
    System.out.println("Raw: " + rawExpression + " ------> Whereclause: " + whereClause);
    System.out.println();
  }

  @Test
  public void printExpressionWithProperty() throws Exception {
    // Use a mocked edmProvider for this tutorial
    TestEdmProvider provider = new TestEdmProvider();
    Edm edm = RuntimeDelegate.createEdm(provider);
    EdmEntityType entityType =
        edm.getEntityType(TestEdmProvider.NAMESPACE_1, TestEdmProvider.ENTITY_TYPE_1_1.getName());

    String rawExpression = "EmployeeId eq '1'";
    FilterExpression expression = UriParser.parseFilter(null, entityType, rawExpression);
    String whereClause = (String) expression.accept(new JdbcSimpleStringVisitor());
    System.out.println("Raw: " + rawExpression + " ------> Whereclause: " + whereClause);
    System.out.println();
  }

  @Test
  public void printSomeStatementsProperty() throws Exception {
    printWithProperty("EmployeeId eq '1'");
    printWithProperty("EmployeeId lt '1'");
    printWithProperty("EmployeeId le '1'");
    printWithProperty("EmployeeId gt '1'");
    printWithProperty("EmployeeId ge '1'");
    printWithProperty("EmployeeId ne '1'");
    printWithProperty("EmployeeId ne '1' or ManagerId lt '2'");
    printWithProperty("EmployeeId ne '1' and ManagerId lt '3'");
  }

  @Test(expected = PropertyNotFoundException.class)
  public void negativeTestPropertyNotFound() throws Exception {
    String rawExpression = "WorkerId eq '1'";
    FilterExpression expression = UriParser.parseFilter(null, null, rawExpression);
    expression.accept(new JdbcSimpleStringVisitor());
  }

  @Test
  public void printSomeStatementsLiteral() throws Exception {
    print("'a' eq 'b'");
    print("'a' lt 'b'");
    print("'a' le 'b'");
    print("'a' gt 'b'");
    print("'a' ge 'b'");
    print("'a' ne 'b'");
    print("'a' ne 'b' or 'b' lt'a'");
    print("'a' ne 'b' and 'a' lt 'b'");
  }

  private void print(final String rawExpression) throws Exception {
    FilterExpression expression = UriParser.parseFilter(null, null, rawExpression);
    String whereClause = (String) expression.accept(new JdbcSimpleStringVisitor());
    System.out.println("Raw: " + rawExpression + " ------> Whereclause: " + whereClause);
    System.out.println();
  }

  private void printWithProperty(final String rawExpression) throws Exception {
    // Use a mocked edmProvider for this tutorial
    TestEdmProvider provider = new TestEdmProvider();
    Edm edm = RuntimeDelegate.createEdm(provider);
    EdmEntityType entityType =
        edm.getEntityType(TestEdmProvider.NAMESPACE_1, TestEdmProvider.ENTITY_TYPE_1_1.getName());

    FilterExpression expression = UriParser.parseFilter(null, entityType, rawExpression);
    String whereClause = (String) expression.accept(new JdbcSimpleStringVisitor());
    System.out.println("Raw: " + rawExpression + " ------> Whereclause: " + whereClause);
    System.out.println();
  }
}
