/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package demo.jdbc.visitor.tests;

import org.apache.olingo.odata2.api.edm.Edm;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.rt.RuntimeDelegate;
import org.apache.olingo.odata2.api.uri.UriParser;
import org.apache.olingo.odata2.api.uri.expression.FilterExpression;
import org.junit.Test;

import demo.jdbc.visitor.JdbcAdvancedStringVisitor;
import demo.jdbc.visitor.JdbcSimpleStringVisitor;
import demo.jdbc.visitor.TestEdmProvider;

public class JdbcAdvancedStringVisitorTest {

  @Test
  public void testBinary() throws Exception {
    print("'a' eq 'b'");
    print("'a' eq 'b' eq true");
    print("true eq ('a' eq 'b')");
    print("true eq (('a' eq 'b') eq ('a' eq 'b'))");
  }

  @Test
  public void testProperty() throws Exception {
    printWithProperty("EmployeeId eq 'a'");
    printWithProperty("EmployeeId eq 'a'");
  }

  @Test
  public void compareSimpleAndAdvancedVisitor() throws Exception {
    String rawExpression = "'a' eq 'b' or ('c' eq 'd' and 'b' eq 'd')";
    FilterExpression expression = UriParser.parseFilter(null, null, rawExpression);

    String whereClauseSimple = (String) expression.accept(new JdbcSimpleStringVisitor());
    String whereClauseAdvanced = (String) expression.accept(new JdbcAdvancedStringVisitor());
    System.out.println("Simple: " + whereClauseSimple + " ------> Advanced: " + whereClauseAdvanced);
  }

  public void print(final String rawExpression) throws Exception {
    FilterExpression expression = UriParser.parseFilter(null, null, rawExpression);
    String whereClause = (String) expression.accept(new JdbcAdvancedStringVisitor());
    System.out.println("Raw: " + rawExpression + " ------> Whereclause: " + whereClause);
    System.out.println();
  }

  public void printWithProperty(final String rawExpression) throws Exception {
    TestEdmProvider provider = new TestEdmProvider();
    Edm edm = RuntimeDelegate.createEdm(provider);
    EdmEntityType entityType =
        edm.getEntityType(TestEdmProvider.NAMESPACE_1, TestEdmProvider.ENTITY_TYPE_1_1.getName());

    FilterExpression expression = UriParser.parseFilter(null, entityType, rawExpression);
    String whereClause = (String) expression.accept(new JdbcAdvancedStringVisitor());
    System.out.println("Raw: " + rawExpression + " ------> Whereclause: " + whereClause);
    System.out.println();
  }
}
