/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package demo.jdbc.visitor;

import java.util.ArrayList;
import java.util.List;

import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmLiteral;
import org.apache.olingo.odata2.api.edm.EdmTyped;
import org.apache.olingo.odata2.api.uri.expression.BinaryExpression;
import org.apache.olingo.odata2.api.uri.expression.BinaryOperator;
import org.apache.olingo.odata2.api.uri.expression.ExpressionVisitor;
import org.apache.olingo.odata2.api.uri.expression.FilterExpression;
import org.apache.olingo.odata2.api.uri.expression.LiteralExpression;
import org.apache.olingo.odata2.api.uri.expression.MemberExpression;
import org.apache.olingo.odata2.api.uri.expression.MethodExpression;
import org.apache.olingo.odata2.api.uri.expression.MethodOperator;
import org.apache.olingo.odata2.api.uri.expression.OrderByExpression;
import org.apache.olingo.odata2.api.uri.expression.OrderExpression;
import org.apache.olingo.odata2.api.uri.expression.PropertyExpression;
import org.apache.olingo.odata2.api.uri.expression.SortOrder;
import org.apache.olingo.odata2.api.uri.expression.UnaryExpression;
import org.apache.olingo.odata2.api.uri.expression.UnaryOperator;

public class JdbcPreparedStatementVisitor implements ExpressionVisitor {

  @Override
  public Object visitFilterExpression(final FilterExpression filterExpression, final String expressionString,
      final Object expression) {
    return expression;
  }

  @Override
  public Object visitBinary(final BinaryExpression binaryExpression, final BinaryOperator operator,
      final Object leftSide, final Object rightSide) {
    // Transform the OData filter operator into an equivalent sql operator
    String sqlOperator = "";
    switch (operator) {
    case EQ:
      sqlOperator = "=";
      break;
    case NE:
      sqlOperator = "<>";
      break;
    case OR:
      sqlOperator = "OR";
      break;
    case AND:
      sqlOperator = "AND";
      break;
    case GE:
      sqlOperator = ">=";
      break;
    case GT:
      sqlOperator = ">";
      break;
    case LE:
      sqlOperator = "<=";
      break;
    case LT:
      sqlOperator = "<";
      break;
    default:
      // Other operators are not supported for SQL Statements
      throw new UnsupportetOperatorException("Unsupported operator: " + operator.toUriLiteral());
    }
    // The idea is to check if the left side is of type property. If this is the case we append the property name and
    // the operator to the where clause
    if (leftSide instanceof EdmTyped && rightSide instanceof String) {
      Expression expression = new Expression(operator);
      try {
        expression.setPrepeparedWhere(((EdmTyped) leftSide).getName() + " " + sqlOperator + " ?");
      } catch (EdmException e) {
        throw new RuntimeException("EdmException occured");
      }
      expression.addParameter(rightSide);
      return expression;
    } else if (leftSide instanceof Expression && rightSide instanceof Expression) {
      Expression returnExpression = new Expression(operator);
      Expression leftSideExpression = (Expression) leftSide;
      if (BinaryOperator.AND.equals(leftSideExpression.getOperator())
          || BinaryOperator.OR.equals(leftSideExpression.getOperator())) {
        leftSideExpression.setPrepeparedWhere("(" + leftSideExpression.toString() + ")");
      }
      Expression rightSideExpression = (Expression) rightSide;
      if (BinaryOperator.AND.equals(rightSideExpression.getOperator())
          || BinaryOperator.OR.equals(rightSideExpression.getOperator())) {
        rightSideExpression.setPrepeparedWhere("(" + rightSideExpression.toString() + ")");
      }
      returnExpression.setPrepeparedWhere(leftSideExpression.toString() + " " + sqlOperator + " "
          + rightSideExpression.toString());

      for (Object parameter : leftSideExpression.getParameters()) {
        returnExpression.addParameter(parameter);
      }
      for (Object parameter : rightSideExpression.getParameters()) {
        returnExpression.addParameter(parameter);
      }
      return returnExpression;
    } else {
      throw new RuntimeException("Not right format");
    }
  }

  @Override
  public Object visitOrderByExpression(final OrderByExpression orderByExpression, final String expressionString,
      final List<Object> orders) {
    throw new NotImplementedException();
  }

  @Override
  public Object visitOrder(final OrderExpression orderExpression, final Object filterResult, final SortOrder sortOrder) {
    throw new NotImplementedException();
  }

  @Override
  public Object visitLiteral(final LiteralExpression literal, final EdmLiteral edmLiteral) {
    // Sql Injection is not possible anymore since we are using prepared statements. Thus we can just give back the
    // edmLiteral content
    return edmLiteral.getLiteral();
  }

  @Override
  public Object visitMethod(final MethodExpression methodExpression, final MethodOperator method,
      final List<Object> parameters) {
    throw new NotImplementedException();
  }

  @Override
  public Object visitMember(final MemberExpression memberExpression, final Object path, final Object property) {
    throw new NotImplementedException();
  }

  @Override
  public Object visitProperty(final PropertyExpression propertyExpression, final String uriLiteral,
      final EdmTyped edmProperty) {
    if (edmProperty == null) {
      // If a property is not found it wont be represented in the database thus we have to throw an exception
      throw new PropertyNotFoundException("Could not find Property: " + uriLiteral);
    } else {
      // To distinguish between literals and properties we give back the whole edmProperty in this case
      return edmProperty;
    }
  }

  @Override
  public Object visitUnary(final UnaryExpression unaryExpression, final UnaryOperator operator, final Object operand) {
    throw new NotImplementedException();
  }

  public class Expression {
    private String preparedWhere;
    private List<Object> parameters;
    private BinaryOperator operator;

    public Expression(final BinaryOperator operator) {
      preparedWhere = "";
      parameters = new ArrayList<Object>();
      this.operator = operator;
    }

    public void addParameter(final Object parameter) {
      parameters.add(parameter);
    }

    public void setPrepeparedWhere(final String where) {
      preparedWhere = where;
    }

    public List<Object> getParameters() {
      return parameters;
    }

    public BinaryOperator getOperator() {
      return operator;
    }

    @Override
    public String toString() {
      return preparedWhere;
    }
  }
}
