/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package demo.jdbc.visitor;

import java.util.List;

import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmLiteral;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeKind;
import org.apache.olingo.odata2.api.edm.EdmTyped;
import org.apache.olingo.odata2.api.uri.expression.BinaryExpression;
import org.apache.olingo.odata2.api.uri.expression.BinaryOperator;
import org.apache.olingo.odata2.api.uri.expression.ExpressionVisitor;
import org.apache.olingo.odata2.api.uri.expression.FilterExpression;
import org.apache.olingo.odata2.api.uri.expression.LiteralExpression;
import org.apache.olingo.odata2.api.uri.expression.MemberExpression;
import org.apache.olingo.odata2.api.uri.expression.MethodExpression;
import org.apache.olingo.odata2.api.uri.expression.MethodOperator;
import org.apache.olingo.odata2.api.uri.expression.OrderByExpression;
import org.apache.olingo.odata2.api.uri.expression.OrderExpression;
import org.apache.olingo.odata2.api.uri.expression.PropertyExpression;
import org.apache.olingo.odata2.api.uri.expression.SortOrder;
import org.apache.olingo.odata2.api.uri.expression.UnaryExpression;
import org.apache.olingo.odata2.api.uri.expression.UnaryOperator;

public class JdbcAdvancedStringVisitor implements ExpressionVisitor {

  @Override
  public Object visitFilterExpression(final FilterExpression filterExpression, final String expressionString,
      final Object expression) {
    return "WHERE " + expression;
  }

  @Override
  public Object visitBinary(final BinaryExpression binaryExpression, final BinaryOperator operator,
      final Object leftSide, final Object rightSide) {
    String actualLeftSide = leftSide.toString();
    String actualRightSide = rightSide.toString();
    if (leftSide instanceof Expression) {
      // If something is lower in the tree and is of the type AND or OR it needs brackets to show the higher priority
      if (BinaryOperator.AND.equals(((Expression) leftSide).getOperator())
          || BinaryOperator.OR.equals(((Expression) leftSide).getOperator())) {
        actualLeftSide = "(" + leftSide + ")";
      }
    }
    if (rightSide instanceof Expression) {
      // If something is lower in the tree and is of the type AND or OR it needs brackets to show the higher priority
      if (BinaryOperator.AND.equals(((Expression) rightSide).getOperator())
          || BinaryOperator.OR.equals(((Expression) rightSide).getOperator())) {
        actualRightSide = "(" + rightSide + ")";
      }
    }
    // Transform the OData filter operator into an equivalent sql operator
    String sqlOperator = "";
    switch (operator) {
    case EQ:
      sqlOperator = "=";
      break;
    case NE:
      sqlOperator = "<>";
      break;
    case OR:
      sqlOperator = "OR";
      break;
    case AND:
      sqlOperator = "AND";
      break;
    case GE:
      sqlOperator = ">=";
      break;
    case GT:
      sqlOperator = ">";
      break;
    case LE:
      sqlOperator = "<=";
      break;
    case LT:
      sqlOperator = "<";
      break;
    default:
      // Other operators are not supported for SQL Statements
      throw new UnsupportetOperatorException("Unsupported operator: " + operator.toUriLiteral());
    }

    // return the binary statement
    return new Expression(actualLeftSide + " " + sqlOperator + " " + actualRightSide, operator);
  }

  @Override
  public Object visitOrderByExpression(final OrderByExpression orderByExpression, final String expressionString,
      final List<Object> orders) {
    throw new NotImplementedException();
  }

  @Override
  public Object visitOrder(final OrderExpression orderExpression, final Object filterResult, final SortOrder sortOrder) {
    throw new NotImplementedException();
  }

  @Override
  public Object visitLiteral(final LiteralExpression literal, final EdmLiteral edmLiteral) {
    if (EdmSimpleTypeKind.String.getEdmSimpleTypeInstance().equals(edmLiteral.getType())) {
      // we have to be carefull with strings due to sql injection
      // /TODO: Prevent sql injection via escaping
      return "'" + edmLiteral.getLiteral() + "'";
    } else {
      return "'" + edmLiteral.getLiteral() + "'";
    }
  }

  @Override
  public Object visitMethod(final MethodExpression methodExpression, final MethodOperator method,
      final List<Object> parameters) {
    throw new NotImplementedException();
  }

  @Override
  public Object visitMember(final MemberExpression memberExpression, final Object path, final Object property) {
    throw new NotImplementedException();
  }

  @Override
  public Object visitProperty(final PropertyExpression propertyExpression, final String uriLiteral,
      final EdmTyped edmProperty) {
    if (edmProperty == null) {
      // If a property is not found it wont be represented in the database thus we have to throw an exception
      throw new PropertyNotFoundException("Could not find Property: " + uriLiteral);
    } else {
      // Use either edm.getProperty(uriLiteral).toString() or an internal HashMap where edmPropertyName is the key and
      // the value is the database name
      // It is also possible to use the mapping of the edmProperty
      try {
        return edmProperty.getName();
      } catch (EdmException e) {
        throw new PropertyNotFoundException(e);
      }
    }
  }

  @Override
  public Object visitUnary(final UnaryExpression unaryExpression, final UnaryOperator operator, final Object operand) {
    throw new NotImplementedException();
  }

  private class Expression {
    private String expressionString;
    private BinaryOperator operator;

    public Expression(final String expressionString, final BinaryOperator operator) {
      this.expressionString = expressionString;
      this.operator = operator;
    }

    public BinaryOperator getOperator() {
      return operator;
    }

    @Override
    public String toString() {
      return expressionString;
    }
  }

}